<?php
require_once __DIR__ . '/../db.php';

class Transaction {
    public static function create(int $user_id, string $title, float $amount, string $type, ?string $category=null): int {
        $conn = db();
        $stmt = $conn->prepare("INSERT INTO transactions(user_id,title,amount,type,category) VALUES(?,?,?,?,?)");
        $stmt->bind_param("isdss", $user_id, $title, $amount, $type, $category);
        $stmt->execute();
        return $stmt->insert_id;
    }

    public static function allForUser(int $user_id): array {
        $conn = db();
        $stmt = $conn->prepare("SELECT id, title, amount, type, category, created_at FROM transactions WHERE user_id=? ORDER BY created_at DESC");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $res = $stmt->get_result();
        return $res->fetch_all(MYSQLI_ASSOC);
    }

    public static function deleteForUser(int $txn_id, int $user_id): bool {
        $conn = db();
        $stmt = $conn->prepare("DELETE FROM transactions WHERE id=? AND user_id=?");
        $stmt->bind_param("ii", $txn_id, $user_id);
        $stmt->execute();
        return $stmt->affected_rows > 0;
    }

    public static function summaryForUser(int $user_id): array {
        $conn = db();
        $stmt = $conn->prepare("
            SELECT 
              SUM(CASE WHEN type='income' THEN amount ELSE 0 END) AS income_total,
              SUM(CASE WHEN type='expense' THEN amount ELSE 0 END) AS expense_total,
              COUNT(*) AS txn_count
            FROM transactions
            WHERE user_id=?
        ");
        $stmt->bind_param("i", $user_id);
        $stmt->execute();
        $res = $stmt->get_result()->fetch_assoc();
        $income = (float)($res['income_total'] ?? 0);
        $expense = (float)($res['expense_total'] ?? 0);
        return [
            'income_total' => $income,
            'expense_total' => $expense,
            'balance' => $income - $expense,
            'txn_count' => (int)($res['txn_count'] ?? 0),
        ];
    }
}
